#!/usr/bin/env node
/**
 * AGA Evidence Bundle Offline Verifier - Sample v1.0.0
 *
 * Usage: node verify.js [bundle-directory]
 * If no directory specified, uses current directory
 */

const fs = require('fs');
const path = require('path');
const crypto = require('crypto');

function sha256(data) {
  return crypto.createHash('sha256').update(data).digest('hex');
}

function verify(bundlePath) {
  console.log('=====================================');
  console.log('AGA Evidence Bundle Offline Verifier');
  console.log('=====================================');
  console.log('Bundle path:', bundlePath);
  console.log('');

  const results = {
    artifact: 'PENDING',
    receipts: 'PENDING',
    chain: 'PENDING',
    drift: 'NO',
    enforcement: 'NONE',
    merkle: 'SKIPPED',
    anchor: 'SKIPPED',
    overall: 'PENDING'
  };

  try {
    // V1: Load manifest
    const manifestPath = path.join(bundlePath, 'bundle_manifest.json');
    if (!fs.existsSync(manifestPath)) {
      throw new Error('bundle_manifest.json not found');
    }
    const manifest = JSON.parse(fs.readFileSync(manifestPath, 'utf8'));

    // V2: Verify file checksums
    let checksumErrors = 0;
    for (const [filePath, meta] of Object.entries(manifest.files)) {
      const fullPath = path.join(bundlePath, filePath);
      if (fs.existsSync(fullPath)) {
        const content = fs.readFileSync(fullPath);
        const digest = sha256(content);
        if (digest !== meta.sha256) {
          console.log('CHECKSUM MISMATCH:', filePath);
          checksumErrors++;
        }
      } else {
        console.log('FILE NOT FOUND:', filePath);
        checksumErrors++;
      }
    }

    // V3: Load policy artifact
    const policyPath = path.join(bundlePath, 'policy', 'policy_artifact.json');
    const policy = JSON.parse(fs.readFileSync(policyPath, 'utf8'));
    console.log('Policy ID:', policy.policy_id.substring(0, 16) + '...');

    // V4: Load chain head
    const chainHeadPath = path.join(bundlePath, 'receipts', 'chain_head.json');
    const chainHead = JSON.parse(fs.readFileSync(chainHeadPath, 'utf8'));

    // V5: Load and verify receipts
    const receiptsDir = path.join(bundlePath, 'receipts');
    const receiptFiles = fs.readdirSync(receiptsDir)
      .filter(f => f.match(/^\\d{4}\\.json$/))
      .sort();

    let chainValid = true;
    let receiptCount = 0;
    let prevHash = '0'.repeat(64);

    for (const file of receiptFiles) {
      const receipt = JSON.parse(fs.readFileSync(path.join(receiptsDir, file), 'utf8'));
      receiptCount++;

      // Check chain linkage
      if (receipt.chain.prev_receipt_hash !== prevHash) {
        console.log('CHAIN BREAK at receipt', receipt.counter);
        chainValid = false;
      }
      prevHash = receipt.chain.this_receipt_hash;

      // Check for drift/enforcement
      if (receipt.event_type === 'DRIFT_DETECTED') {
        results.drift = 'YES (receipt #' + receipt.counter + ', reason: ' + receipt.decision.reason_code + ')';
      }
      if (receipt.event_type === 'ENFORCED') {
        results.enforcement = receipt.decision.action + ' (receipt #' + receipt.counter + ')';
      }
    }

    // V6: Verify chain head
    if (chainHead.head_receipt_hash !== prevHash) {
      console.log('CHAIN HEAD MISMATCH');
      chainValid = false;
    }
    if (chainHead.receipt_count !== receiptCount) {
      console.log('RECEIPT COUNT MISMATCH');
      chainValid = false;
    }

    // Compile results
    results.artifact = checksumErrors === 0 ? 'OK' : 'FAIL (' + checksumErrors + ' errors)';
    results.receipts = 'OK (' + receiptCount + '/' + receiptCount + ')';
    results.chain = chainValid ? 'OK' : 'FAIL';
    results.overall = (checksumErrors === 0 && chainValid) ? 'PASS' : 'FAIL';

    // Print results
    console.log('');
    console.log('=== VERIFICATION RESULTS ===');
    console.log('artifact:', results.artifact);
    console.log('receipts:', results.receipts);
    console.log('chain:', results.chain);
    console.log('drift:', results.drift);
    console.log('enforcement:', results.enforcement);
    console.log('merkle:', results.merkle);
    console.log('anchor:', results.anchor);
    console.log('');
    console.log('overall:', results.overall);

    process.exit(results.overall === 'PASS' ? 0 : 1);

  } catch (error) {
    console.error('Verification error:', error.message);
    results.overall = 'FAIL';
    process.exit(1);
  }
}

const bundlePath = process.argv[2] || '.';
verify(bundlePath);
